#include "process.h"
#include "frame.h"
#include "symtab.h"
#include "symbol.h"
#include "core.h"
#include "coff.h"
#include "asm.h"
#include "mac32asm.h"
#include "mac32core.h"
SRCFILE("mac32core.c")

const char	CALL_INST = 0x2c,
		RET_INST = 0x08,
		ADDW2_INST = 0x9C,
		SAVE_INST = 0x10;	/* SAVE Rn saves n..LASTREG */

int Mac32Core::REG_FP()			{ return 9; }
int Mac32Core::REG_AP()			{ return 10; }
int Mac32Core::REG_SP()			{ return 12; }
int Mac32Core::REG_PC()			{ return 13; }
int Mac32Core::nregs()			{ return 16; }
long Mac32Core::returnregloc()		{ return regloc(0); }
long Mac32Core::callingpc(long fp)	{ return peek(fp-36)->lng; }
long Mac32Core::callingfp(long fp)	{ return peek(fp-28)->lng; }
int Mac32Core::atjsr(long pc)		{ return peek(pc)->chr == CALL_INST; }
int Mac32Core::atreturn(long pc)	{ return peek(pc)->chr == RET_INST; }
Asm *Mac32Core::newAsm()		{ return new Mac32Asm(this); }

void Mac32Core::newSymTab(long reloc)
{
	_symtab = new CoffSymTab(this, stabfd, _symtab, reloc);
}

Mac32Core::Mac32Core()
{
	stackdir = GROWUP;
	memlayout = MSBFIRST;
	bptsize = 1;
}

char *Mac32Core::regname(int r)
{
	switch( r ){
	case  9:	return "fp";
	case 10:	return "ap";
	case 11:	return "psw";
	case 12:	return "sp";
	case 13:	return "pc";
	case 14:	return "pcbp";
	case 15:	return "isp";
	default:	return sf( "r%d", r );
	}
}

#define LASTREG 8
long Mac32Core::saved(Frame *f, int r, int)	/* search symtab, then RAM */
{
	Func *func;

	if( !f || !f->regbase ) return 0;
	if( !(func = f->func) ){
		f->addsymbols();
		if( !(func = f->func) ) return 0;
	}
	if( func->regsave == -1 ){
		if( peek(func->range.lo)->chr != SAVE_INST )
			func->regsave = LASTREG+1;
		else
			func->regsave = 0xF & peek(func->range.lo+1)->chr;
	}
	if( r>LASTREG || r<func->regsave || !f->regbase )
		return 0;
	return f->regbase+(r-func->regsave)*4;
}

Frame Mac32Core::frameabove(long _fp)
{
	Frame f(this);
	long x[3];
	if( _fp ){
		read(_fp-36, (char *)x, sizeof(x));
		f.pc = x[0];
		f.ap = x[1];
		f.fp = x[2];
	} else {
		f.pc = pc();
		f.fp = fp();
		f.ap = ap();
	}
	f.regbase = f.fp-24;
	return f;
}

long Mac32Core::instrafterjsr()	
{
	dostep(0,0,1);
	return peek(sp()-8)->lng;
}

char *Mac32Core::stepprolog()
{
	char *error = 0;
	if( peek(pc())->chr == SAVE_INST )
		error = step();
	if( !error &&
	    (unsigned char)peek(pc())->chr == (unsigned char)ADDW2_INST )
		error = step();
	return error;
}

char *Mac32Core::docall(long addr, int numarg)
{
	const int CALL_SIZE=8;
	char buildcall[CALL_SIZE], save[CALL_SIZE];
	char *error;

	if( behavetype() == ACTIVE )
		return "process not stopped";
	buildcall[0] = CALL_INST;
	buildcall[1] = 0xCC;	// xx(%sp)
	buildcall[2] = -4 * numarg;
	buildcall[3] = 0x7F;	// Absolute address
	buildcall[4] = char(addr & 0xFF);
	buildcall[5] = char((addr >> 8) & 0xFF);
	buildcall[6] = char((addr >> 16) & 0xFF);
	buildcall[7] = char((addr >> 24) & 0xFF);
	int callstart = (int)scratchaddr();
	if( ( error = read(callstart, save ,CALL_SIZE) )
	 || ( error = write(callstart, buildcall, CALL_SIZE) )
	 || ( error = regpoke(REG_PC(), callstart) )
	 || ( error = step( callstart, callstart+CALL_SIZE) )
	 || ( error = write(callstart, save, CALL_SIZE) ) )
		return error;
	return 0;
}

long Mac32Core::apforcall(int argbytes)
{
	long ap = sp();
	regpoke(REG_SP(), sp() + argbytes);
	return ap;
}
